<?php

namespace App\Http\Controllers;

use App\Models\Arsip;
use App\Models\ActivityLog;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class ArsipController extends Controller
{
    /* =========================================================
     | ADMIN – ARSIP AKTIF
     ========================================================= */
    public function aktif(Request $request)
    {
        $query = Arsip::where('status', 'aktif');

        if ($request->filled('tahun_awal')) {
            $query->where('kurun_waktu', '>=', $request->tahun_awal);
        }

        if ($request->filled('tahun_akhir')) {
            $query->where('kurun_waktu', '<=', $request->tahun_akhir);
        }

        if ($request->filled('kode')) {
            $query->where('kode_klasifikasi', 'like', "%{$request->kode}%");
        }

        if ($request->filled('indeks')) {
            $query->where('indeks', 'like', "%{$request->indeks}%");
        }

        if ($request->filled('uraian')) {
            $query->where('uraian', 'like', "%{$request->uraian}%");
        }

        if ($request->print === 'true') {
            return view('admin.arsip.print', [
                'arsips' => $query->orderBy('kurun_waktu')->get()
            ]);
        }

        $arsips = $query->orderBy('kurun_waktu', 'desc')
                        ->paginate(10)
                        ->withQueryString();

        return view('admin.arsip.aktif', compact('arsips'));
    }

    /* =========================================================
     | ADMIN – STATUS
     ========================================================= */
    public function inaktifkan($id)
    {
        Arsip::where('id', $id)->update(['status' => 'inaktif']);
        return back();
    }

    public function aktifkan($id)
    {
        Arsip::where('id', $id)->update(['status' => 'aktif']);
        return back();
    }

    public function hapus($id)
    {
        Arsip::where('id', $id)->update(['status' => 'terhapus']);
        return back();
    }

    /* =========================================================
     | ADMIN – ARSIP INAKTIF
     ========================================================= */
    public function inaktif(Request $request)
    {
        $query = Arsip::where('status', 'inaktif');

        if ($request->filled('tahun_awal')) {
            $query->where('kurun_waktu', '>=', $request->tahun_awal);
        }

        if ($request->filled('tahun_akhir')) {
            $query->where('kurun_waktu', '<=', $request->tahun_akhir);
        }

        if ($request->filled('kode')) {
            $query->where('kode_klasifikasi', 'like', "%{$request->kode}%");
        }

        if ($request->filled('indeks')) {
            $query->where('indeks', 'like', "%{$request->indeks}%");
        }

        if ($request->filled('uraian')) {
            $query->where('uraian', 'like', "%{$request->uraian}%");
        }

        $arsips = $query->orderBy('kurun_waktu', 'desc')
                        ->paginate(10)
                        ->withQueryString();

        return view('admin.arsip.inaktif', compact('arsips'));
    }

    /* =========================================================
     | ADMIN – ARSIP TERHAPUS
     ========================================================= */
    public function terhapus(Request $request)
    {
        $query = Arsip::where('status', 'terhapus');

        if ($request->filled('tahun_awal')) {
            $query->where('kurun_waktu', '>=', $request->tahun_awal);
        }

        if ($request->filled('tahun_akhir')) {
            $query->where('kurun_waktu', '<=', $request->tahun_akhir);
        }

        $arsips = $query->orderBy('kurun_waktu', 'desc')
                        ->paginate(10)
                        ->withQueryString();

        return view('admin.arsip.terhapus', compact('arsips'));
    }

    /* =========================================================
     | ADMIN – CREATE & UPDATE
     ========================================================= */
    public function create()
    {
        return view('admin.arsip.create');
    }

    public function store(Request $request)
    {
        $data = $request->validate([
            'kode_klasifikasi'     => 'required',
            'indeks'               => 'required',
            'uraian'               => 'required',
            'kurun_waktu'          => 'required|numeric',
            'tingkat_perkembangan' => 'required',
            'jumlah'               => 'required',
            'keterangan'           => 'nullable',
            'nomor_definitif'      => 'required',
            'nomor_boks'           => 'required',
            'lokasi_simpan'        => 'required',
            'file_pdf'             => 'nullable|mimes:pdf|max:2048'
        ]);

        $data['status'] = 'aktif';

        if ($request->hasFile('file_pdf')) {
            $nama = time().'_'.$request->file('file_pdf')->getClientOriginalName();
            $request->file('file_pdf')->move(public_path('arsip'), $nama);
            $data['file_pdf'] = 'arsip/'.$nama;
        }

        Arsip::create($data);
        return redirect()->route('arsip.aktif')->with('success', 'Arsip berhasil ditambahkan');
    }

    public function edit($id)
    {
        $arsip = Arsip::findOrFail($id);
        return view('admin.arsip.edit', compact('arsip'));
    }

    public function update(Request $request, $id)
    {
        $arsip = Arsip::findOrFail($id);

        if ($request->hasFile('file_pdf')) {
            $nama = time().'_'.$request->file('file_pdf')->getClientOriginalName();
            $request->file('file_pdf')->move(public_path('arsip'), $nama);
            $arsip->file_pdf = 'arsip/'.$nama;
        }

        $arsip->update($request->except('file_pdf'));
        return back()->with('success', 'Data arsip diperbarui');
    }

    /* =========================================================
     | LIHAT FILE + LOG
     ========================================================= */
    public function lihatFile($id)
    {
        $arsip = Arsip::findOrFail($id);

        if (!$arsip->file_pdf || !file_exists(public_path($arsip->file_pdf))) {
            abort(404);
        }

        ActivityLog::create([
            'user_id'   => Auth::id(),
            'aktivitas' => Auth::user()->username.' melihat file arsip "'.$arsip->uraian.'"',
            'ip_address'=> request()->ip(),
        ]);

        return response()->file(public_path($arsip->file_pdf));
    }

    /* =========================================================
     | USER – ARSIP AKTIF (READ ONLY)
     ========================================================= */
    public function aktifUser(Request $request)
    {
        $query = Arsip::where('status', 'aktif');

        if ($request->filled('q')) {
            $query->where(function ($q) use ($request) {
                $q->where('kode_klasifikasi', 'like', "%{$request->q}%")
                  ->orWhere('indeks', 'like', "%{$request->q}%")
                  ->orWhere('uraian', 'like', "%{$request->q}%");
            });
        }

        $arsips = $query->orderBy('kurun_waktu', 'desc')
                        ->paginate(10)
                        ->withQueryString();

        return view('user.arsip.aktif', compact('arsips'));
    }

    /* =========================================================
     | USER – ARSIP INAKTIF (READ ONLY)
     ========================================================= */
    public function inaktifUser(Request $request)
    {
        $query = Arsip::where('status', 'inaktif');

        if ($request->filled('kode')) {
            $query->where('kode_klasifikasi', 'like', "%{$request->kode}%");
        }

        if ($request->filled('indeks')) {
            $query->where('indeks', 'like', "%{$request->indeks}%");
        }

        if ($request->filled('uraian')) {
            $query->where('uraian', 'like', "%{$request->uraian}%");
        }

        $arsips = $query->orderBy('kurun_waktu', 'desc')
                        ->paginate(10)
                        ->withQueryString();

        return view('user.arsip.inaktif', compact('arsips'));
    }
}
